const express = require('express');
const multer = require('multer');
const DXFParser = require('dxf-parser');
const fs = require('fs');
const path = require('path');

const app = express();
const upload = multer({ dest: 'uploads/' });

app.use(express.static('public'));

app.post('/upload', upload.single('dxfFile'), (req, res) => {
  if (!req.file) return res.status(400).json({ error: 'No file uploaded' });

  const parser = new DXFParser();
  const data = fs.readFileSync(req.file.path, 'utf-8');

  let parsed = [];
  try {
    const dxf = parser.parseSync(data);
    parsed = (dxf.entities || []).filter(e => {
      return e.type === 'LWPOLYLINE' || e.type === 'POLYLINE' || e.type === 'POLYLINE3D';
    }).map(e => {
      // Normalize vertices for different entity structures
      let vertices = [];
      if (e.vertices && Array.isArray(e.vertices)) {
        vertices = e.vertices.map(v => ({ x: v.x, y: v.y }));
      } else if (e.points && Array.isArray(e.points)) {
        vertices = e.points.map(v => ({ x: v.x, y: v.y }));
      }
      return {
        layer: e.layer || 'UNKNOWN',
        points: vertices
      };
    });
  } catch (err) {
    fs.unlinkSync(req.file.path);
    return res.status(500).json({ error: 'Failed to parse DXF', details: err.message });
  } finally {
    // remove temp file if still exists
    try { if (fs.existsSync(req.file.path)) fs.unlinkSync(req.file.path); } catch(e){}
  }

  // Simple problem analysis
  const result = parsed.map(p => {
    let status = 'Pass';
    let issue = '-';
    if (!p.points || p.points.length < 3) { status='Warning'; issue='Too few points'; }
    const first = p.points && p.points[0];
    const last = p.points && p.points[p.points.length-1];
    if (first && last) {
      const dx = last.x - first.x;
      const dy = last.y - first.y;
      if (Math.sqrt(dx*dx + dy*dy) > 1e-6 && Math.sqrt(dx*dx + dy*dy) > 5) { status='Error'; issue='Open polyline'; }
    } else {
      status = 'Warning'; issue = 'No vertices found';
    }
    return { pattern: p.layer, status, issue, points: p.points || [] };
  });

  res.json(result);
});

const PORT = process.env.PORT || 3000;
app.listen(PORT, () => console.log(`Server running at http://localhost:${PORT}`));
