const input = document.getElementById('dxfInput');
const uploadBtn = document.getElementById('uploadBtn');
const tableBody = document.querySelector('#resultTable tbody');
const canvas = document.getElementById('canvas');
const ctx = canvas.getContext('2d');

uploadBtn.addEventListener('click', async () => {
  if (!input.files[0]) { alert('Select a DXF file first'); return; }
  uploadBtn.disabled = true;
  uploadBtn.textContent = 'Analyzing...';

  const fd = new FormData();
  fd.append('dxfFile', input.files[0]);

  try {
    const res = await fetch('/upload', { method: 'POST', body: fd });
    if (!res.ok) {
      const err = await res.json().catch(()=>({error:'Unknown'}));
      alert('Server error: ' + (err.error || JSON.stringify(err)));
      uploadBtn.disabled = false;
      uploadBtn.textContent = 'Upload & Analyze';
      return;
    }
    const data = await res.json();
    renderTable(data);
    drawCanvas(data);
  } catch (e) {
    alert('Network or unexpected error: ' + e.message);
  } finally {
    uploadBtn.disabled = false;
    uploadBtn.textContent = 'Upload & Analyze';
  }
});

function renderTable(patterns) {
  tableBody.innerHTML = '';
  patterns.forEach(p => {
    const tr = document.createElement('tr');
    tr.innerHTML = `<td>${p.pattern}</td><td class="status-${p.status}">${p.status}</td><td>${p.issue}</td>`;
    tableBody.appendChild(tr);
  });
}

function drawCanvas(patterns) {
  ctx.clearRect(0,0,canvas.width,canvas.height);
  // compute bounding box to scale nicely
  let minX=Infinity,minY=Infinity,maxX=-Infinity,maxY=-Infinity;
  patterns.forEach(p => {
    p.points.forEach(pt=>{
      if(pt.x<minX) minX=pt.x;
      if(pt.y<minY) minY=pt.y;
      if(pt.x>maxX) maxX=pt.x;
      if(pt.y>maxY) maxY=pt.y;
    });
  });
  if (minX===Infinity) return;
  const padding = 20;
  const w = maxX - minX || 1;
  const h = maxY - minY || 1;
  const scale = Math.min((canvas.width-2*padding)/w, (canvas.height-2*padding)/h);
  patterns.forEach(p=>{
    if(!p.points.length) return;
    ctx.beginPath();
    ctx.lineWidth = 2;
    ctx.strokeStyle = p.status==='Pass'?'#16a34a':(p.status==='Warning'?'#d97706':'#dc2626');
    p.points.forEach((pt,i)=>{
      const x = padding + (pt.x - minX)*scale;
      const y = canvas.height - (padding + (pt.y - minY)*scale);
      if(i===0) ctx.moveTo(x,y); else ctx.lineTo(x,y);
    });
    ctx.stroke();
    // label at centroid
    const cx = p.points.reduce((s,pt)=>s+pt.x,0)/p.points.length;
    const cy = p.points.reduce((s,pt)=>s+pt.y,0)/p.points.length;
    const lx = padding + (cx - minX)*scale;
    const ly = canvas.height - (padding + (cy - minY)*scale);
    ctx.fillStyle = '#0f172a';
    ctx.font = '12px sans-serif';
    ctx.fillText(p.pattern, lx+6, ly+6);
  });
}
